/*----------------------------------------------------------

Gauss Program used in the paper

"A joint model for the term structure of interest rates and the macroeconomy"
forthcoming in the Journal of Applied Econometrics

by

Hans Dewachter   (hans.dewachter@econ.kuleuven.ac.be)
Marco Lyrio      (marco.lyrio@econ.kuleuven.ac.be)
Konstantijn Maes (Konstantijn.Maes@nbb.be)

Forthcoming in the Journal of Applied Econometrics

November 2004
----------------------------------------------------------*/

new; cls;
library cml pgraph;
#include cml.ext;
cmlset;
graphset;
format /m1 /rd 16,8;
outwidth 200;

@*********************************************************@
@********************   SETTINGS   ***********************@
@*********************************************************@

filedir =  "c:\\jae\\";
filedata = "c:\\jae\\";


file01 = filedata $+ "data.txt";
file02 = filedir  $+ "bestever.txt";
file03 = filedir  $+ "bestbnew.txt";


@------------------------------------------------------------@
filebestb = 0; @ 0:read initial set of parameters from below;
                 1:read from file "bestbnew"                 @
@------------------------------------------------------------@
ns        =   5; @ No. of states/factors @
nmData    =   6; @ No. of maturities in the data file @
nosData   =   2; @ No. of observable states in the data file (2= output and inflation) @
interval  =   2; @ Data, 1: monthly, 2:quarterly @
ndTot     = 164; @ Total no. of points in the dataset @
ndInit    =  25; @ Initial point included in the estimation @
ndLast    = 164; @ Last point included in the estimation    @

@ Maturities in the dataset (in quarters) @ @ 0:Data NOT included in the estimation; 1:INCLUDED @
let matdata  = 1   2   4   8  20  40;          @ Maturities in the dataset (in quarters) @
let dataIncl = 1   1   1   1   1   1   1   1 ;  @ 0:Data NOT included in the estimation; 1:INCLUDED @

@--- WHAT SHOULD BE DONE? -----------------------------------------------@
scaleVAR     = 1;   @ 0:skip automatic scaling; 1:scale all variables      @
notstable    = 1;   @ 0=normal; 1=does not require stability of the system @

@-----------------------------------------------------------------------------------@
flagsimann =  0;     @ 0:cml procedure ; 1:simulated annealing procedure (SIMANN) @
	continueopt = 1; @ 0:stop after SIMANN; 1:do CML after SIMANN                 @
	nnt    = 20;     @ no. of iterations before reduction in parameter T (tpar)   @
	frt    = 0.1;    @ 0.85, reduction factor for parameter T (tpar)                    @
	nns    = 10;     @ no. of iterations before step length (VM) is adjusted      @
	neps   = 4;      @ no. of final function values to decide upon termination    @
	errps  = 1e-4;   @ error tolerance for termination                            @
	maxevl = 3000;  @ maximum no. of function evaluations                        @
	tpar   = 10; @ parameter T                                                    @
	scalevm = 0.01;
@-----------------------------------------------------------------------------------@

printCML    = 1000; @ 0:never prints any results; Greater than zero otherwise        @
showMessage =   0; @ 0:no; 1:print all messages that lead to lnlik=-large; goto ssd @
@-----------------------------------------------------------------------------------@
nd=zeros(1,1);nos=zeros(1,1);nm=zeros(1,1);
    nd = ndlast-(ndinit-1);           @ No. of points included in the estimation @
    nm =  sumc(dataIncl[1:nmData,1]); @ No. of maturities included in the estimation @
    nos = sumc(dataIncl[nmData+1:nmData+nosData,1]); @ No. of obs. factors included in the estimation @
@-----------------------------------------------------------------------------------------------------@
maxns  = 7; @ Maximum no. of factors @
maxnm  = 6; @ Maximum no. of maturities @
maxnos = 2; @ Maximum no. of observable factors @
kalln = ns*4 +7 +ns + (1+ns) + (maxnm+maxnos)*(maxnm+maxnos) +4*ns*ns +ns;

@---------------------- FLAGS: Variables included in the CML procedure ---------------------------@
flaa = zeros(maxns,4); flab = zeros(7,1); flxb = zeros(maxns,1);     flde = zeros(1+maxns,1);
flhh = zeros(maxnm+maxnos,maxnm+maxnos);  flmm = zeros(maxns,maxns); flmu = zeros(maxns,maxns);
flee = zeros(maxns,maxns);                flgs = zeros(maxns,1);     flba = zeros(maxns,maxns);
@-------------------------------------------------------------------------------------------------@

@      Kappa            Theta           Lambda                c  @
flaa[1,1]= 0;    flaa[1,2]= 0;    flaa[1,3]= 1;    flaa[1,4]= 1;
flaa[2,1]= 0;    flaa[2,2]= 0;    flaa[2,3]= 1;    flaa[2,4]= 1;
flaa[3,1]= 0;    flaa[3,2]= 0;    flaa[3,3]= 1;    flaa[3,4]= 1;
flaa[4,1]= 0;    flaa[4,2]= 1;    flaa[4,3]= 1;    flaa[4,4]= 1;
flaa[5,1]= 0;    flaa[5,2]= 1;    flaa[5,3]= 0;    flaa[5,4]= 1;
flaa[6,1]= 0;    flaa[6,2]= 0;    flaa[6,3]= 0;    flaa[6,4]= 0;
flaa[7,1]= 0;    flaa[7,2]= 0;    flaa[7,3]= 0;    flaa[7,4]= 0;

flmm[1,1]= 1;    flmm[1,2]= 1;    flmm[1,3]= 1;    flmm[1,4]= 0;    flmm[1,5]= 0;
flmm[2,1]= 1;    flmm[2,2]= 1;    flmm[2,3]= 1;    flmm[2,4]= 0;    flmm[2,5]= 0;
flmm[3,1]= 1;    flmm[3,2]= 1;    flmm[3,3]= 1;    flmm[3,4]= 0;    flmm[3,5]= 0;
flmm[4,1]= 0;    flmm[4,2]= 0;    flmm[4,3]= 0;    flmm[4,4]= 1;    flmm[4,5]= 0;
flmm[5,1]= 0;    flmm[5,2]= 0;    flmm[5,3]= 0;    flmm[5,4]= 0;    flmm[5,5]= 1;

flmu[1,1]= 0;    flmu[1,2]= 0;    flmu[1,3]= 0;    flmu[1,4]= 0;    flmu[1,5]= 0;
flmu[2,1]= 0;    flmu[2,2]= 0;    flmu[2,3]= 0;    flmu[2,4]= 0;    flmu[2,5]= 0;
flmu[3,1]= 1;    flmu[3,2]= 1;    flmu[3,3]= 1;    flmu[3,4]= 1;    flmu[3,5]= 1;
flmu[4,1]= 0;    flmu[4,2]= 0;    flmu[4,3]= 0;    flmu[4,4]= 0;    flmu[4,5]= 0;
flmu[5,1]= 0;    flmu[5,2]= 0;    flmu[5,3]= 0;    flmu[5,4]= 0;    flmu[5,5]= 0;

ii=1;
do while ii le nm;
	ij=1;
	do while ij le nm;
		if ii ge ij;
			flhh[ii,ij]=1;
		endif;
	ij=ij+1;
	endo;
ii=ii+1;
endo;

@-------------------------------------------------------------------------------------------------@
baa=zeros(maxns,5);  bab=zeros(7,1);  bxb=zeros(maxns,1);     bde=zeros(1+maxns,1);
bhh=zeros(maxnm+maxnos,maxnm+maxnos); bmm=zeros(maxns,maxns); bmu=zeros(maxns,maxns);
bee=zeros(maxns,maxns);               bgs=zeros(maxns,1);     bba=zeros(maxns,maxns);

@------------------------- Reduce size of matrices to the minumum needed -------------------------@

baa=baa[1:ns,1:4];   flaa = flaa[1:ns,1:4];
bxb=bxb[1:ns,1];     flxb = flxb[1:ns,1];
bde=bde[1:1+ns,1];   flde = flde[1:1+ns,1];
bmm=bmm[1:ns,1:ns];  flmm = flmm[1:ns,1:ns];
bmu=bmu[1:ns,1:ns];  flmu = flmu[1:ns,1:ns];
bee=bee[1:ns,1:ns];  flee = flee[1:ns,1:ns];
bgs=bgs[1:ns,1];     flgs = flgs[1:ns,1];
bba=bba[1:ns,1:ns];  flba = flba[1:ns,1:ns];

@---------------------------------------- INITIAL VALUES -----------------------------------------@
if filebestb eq 0;

@bestEVERlnlik/nd =  42.01206091785394   @
@c:\jae\@

@     Kappa               Theta                Lambda               c                    d  @
    let baa[5,4]= 
    0.00000000000000     0.00000000000000   -63.99506191875781     0.00027918225861 
    0.00000000000000     0.00000000000000    34.83094103773200     0.00014633539212 
    0.00000000000000     0.00000000000000    32.53919692099355     0.00154487571159 
    0.00000000000000     0.02244285947582   -21.95625224476752     0.00006677264179 
    0.00000000000000     0.01371177667655     0.00000000000000     0.00025349992389 ;

@    Deltay         Ksiy           Deltapi        ksipi          eta4           eta5          inityst  @
    let bab[7,1]= 
  0.0000000000   0.0000000000   0.0000000000   0.0000000000   0.0000000000   0.0000000000   0.0000000000  ;

@  Beta1              Beta2              Beta3              Beta4              Beta5              Beta6              Beta7@
    let bxb[5,1]= 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 ;

@   Delta0  @
@   Delta1            Delta2             Delta3             Delta4             Delta5             Delta6             Delta7@
    let bde[6,1]= 
  0.00000000000000 
  0.00000000000000   1.00000000000000   1.00000000000000   0.00000000000000   0.00000000000000 ;

@ h1                 h2                 h3                 h4                 h5                 h6                 h7                 h8@
bhh[ 1 , 1 ]=  0.00000214989522 ;
bhh[ 1 , 2 ]=  0.00000004551326 ;
bhh[ 1 , 3 ]= -0.00000092029964 ;
bhh[ 1 , 4 ]=  0.00000062894707 ;
bhh[ 1 , 5 ]=  0.00000042145139 ;
bhh[ 1 , 6 ]=  0.00000002401937 ;
bhh[ 2 , 1 ]=  0.00000004551326 ;
bhh[ 2 , 2 ]=  0.00000096801529 ;
bhh[ 2 , 3 ]=  0.00000002638953 ;
bhh[ 2 , 4 ]= -0.00000011081198 ;
bhh[ 2 , 5 ]= -0.00000044717724 ;
bhh[ 2 , 6 ]=  0.00000000541536 ;
bhh[ 3 , 1 ]= -0.00000092029964 ;
bhh[ 3 , 2 ]=  0.00000002638953 ;
bhh[ 3 , 3 ]=  0.00000078922121 ;
bhh[ 3 , 4 ]= -0.00000002343598 ;
bhh[ 3 , 5 ]= -0.00000030426537 ;
bhh[ 3 , 6 ]= -0.00000001701949 ;
bhh[ 4 , 1 ]=  0.00000062894707 ;
bhh[ 4 , 2 ]= -0.00000011081198 ;
bhh[ 4 , 3 ]= -0.00000002343598 ;
bhh[ 4 , 4 ]=  0.00000042736173 ;
bhh[ 4 , 5 ]=  0.00000017814808 ;
bhh[ 4 , 6 ]=  0.00000016843153 ;
bhh[ 5 , 1 ]=  0.00000042145139 ;
bhh[ 5 , 2 ]= -0.00000044717724 ;
bhh[ 5 , 3 ]= -0.00000030426537 ;
bhh[ 5 , 4 ]=  0.00000017814808 ;
bhh[ 5 , 5 ]=  0.00000085003696 ;
bhh[ 5 , 6 ]= -0.00000002125349 ;
bhh[ 6 , 1 ]=  0.00000002401937 ;
bhh[ 6 , 2 ]=  0.00000000541536 ;
bhh[ 6 , 3 ]= -0.00000001701949 ;
bhh[ 6 , 4 ]=  0.00000016843153 ;
bhh[ 6 , 5 ]= -0.00000002125349 ;
bhh[ 6 , 6 ]=  0.00000048826046 ;
@  M1                 M2                 M3                 M4                 M5                 M6                 M7@
  let bmm[5,5]= 
 -0.31458886157626  -1.07481947388211  -0.45545425066722   0.00000000000000   0.00000000000000 
  0.38533772614971  -0.24519025829035  -0.13192398428119   0.00000000000000   0.00000000000000 
 -0.06852356409153  -5.15754676675129  -5.30348711555931   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000  -0.00358168372858   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000  -0.48485526737325 ;

@  Mu1                Mu2                Mu3                Mu4                Mu5                Mu6                Mu7@
  let bmu[5,5]= 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00816522861976  -0.36724779268039  -0.98493951286383  -1.18847668590533  -1.53176451262789 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 ;

@  e1                 e2                 e3                 e4                 e5                 e6                 e7@
  let bee[5,5]= 
  1.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   1.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   1.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   1.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   1.00000000000000 ;

@                  g1     g2     g3     g4     g5     g6     g7@
  let bgs[5,1]=   0.00   0.00   0.00   0.00   0.00 ;

@  beta1              beta2              beta3              beta4              beta5              beta6              beta7@
  let bba[5,5]= 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 
  0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000   0.00000000000000 ;

endif;

@-------------------------------------------------------------------------------------------------@


@-------------------------------------------- CML ------------------------------------------------@
KKEY = ZEROS(1,1); KKEY=KEY;   
_cml_Algorithm =  1;
_cml_LineSearch = 3;
_cml_CovPar =     3;
_cml_DirTol =  1e-6;
_cml_MaxIters = 100000;
_cml_MaxIters = 1;

@----------------------------------- Bounds for CML procedure ------------------------------------@

Lbaa = zeros(ns,4); Ubaa = zeros(ns,4);
Lbaa[.,1] =   -5.0*ones(ns,1);    Ubaa[.,1] =    5.0*ones(ns,1);    @ Kappa @
Lbaa[.,2] =   -0.5*ones(ns,1);    Ubaa[.,2] =    5.0*ones(ns,1);    @ Theta @
Lbaa[.,3] = -100000.0*ones(ns,1); Ubaa[.,3] =  100000.0*ones(ns,1); @ Lambda @
Lbaa[.,4] =    0.0*ones(ns,1);    Ubaa[.,4] =    0.1*ones(ns,1);    @ c @

LUbab = zeros(7,2);
LUbab[.,1] = -2.0*ones(7,1);
LUbab[.,2] =  15000.0*ones(7,1);

LUbxb = zeros(ns,2);
LUbxb[.,1] = -50.0*ones(ns,1);
LUbxb[.,2] =  50.0*ones(ns,1);

LUbde = zeros(1+ns,2);
LUbde[.,1] = -2.0*ones(1+ns,1);
LUbde[.,2] =  2.0*ones(1+ns,1);

Lbhh = zeros(maxnm+maxnos,maxnm+maxnos); Ubhh = zeros(maxnm+maxnos,maxnm+maxnos);
Lbhh = -0.001*ones(maxnm+maxnos,maxnm+maxnos); Lbhh = Lbhh - Lbhh[1,1]*eye(maxnm+maxnos);
Lbhh = Lbhh + 1e-8*eye(maxnm+maxnos);

Ubhh =  0.001*ones(maxnm+maxnos,maxnm+maxnos);

Lbmm = zeros(ns,ns); Ubmm = zeros(ns,ns);
Lbmm = -10*ones(ns,ns); 
Ubmm =  10*ones(ns,ns);

Ubmm[4,4]=0.0;
Ubmm[5,5]=0.0;

Lbmu = zeros(ns,ns); Ubmu = zeros(ns,ns);
Lbmu = -100*ones(ns,ns); 
Ubmu =  100*ones(ns,ns);

Lbee = zeros(ns,ns); Ubee = zeros(ns,ns);
Lbee = -5.0*ones(ns,ns); 
Ubee =  5.0*ones(ns,ns);

LUbgs = zeros(ns,2);
LUbgs[.,1] = -5.0*ones(ns,1);
LUbgs[.,2] =  5.0*ones(ns,1);

Lbba = zeros(ns,ns); Ubba = zeros(ns,ns);
Lbba = -50*ones(ns,ns); 
Ubba =  50*ones(ns,ns);

@-------------------------------------- NAMES of Variables ---------------------------------------@
naa = zeros(ns,4);
nab = zeros(7,1);
nxb = zeros(ns,1);
nde = zeros(1+ns,1);
nhh = zeros(maxnm+maxnos,maxnm+maxnos);
nmm = zeros(ns,ns);
nmu = zeros(ns,ns);
nee = zeros(ns,ns);
ngs = zeros(ns,1);
nba = zeros(ns,ns);

ii=1;
do while ii le ns;
    ij=1;
    do while ij le 4;
        if ij eq 1; fmat ="kappa%*.*lf";  naa[ii,ij] = ftos(ii, fmat, 1,0); endif;
        if ij eq 2; fmat ="theta%*.*lf";  naa[ii,ij] = ftos(ii, fmat, 1,0); endif;
        if ij eq 3; fmat ="lambda%*.*lf"; naa[ii,ij] = ftos(ii, fmat, 1,0); endif;
        if ij eq 4; fmat ="c%*.*lf";      naa[ii,ij] = ftos(ii, fmat, 1,0); endif;
    ij=ij+1;
    endo;

    ij=1;
    do while ij le ns;
        fmat ="%*.*lf";      mbz= ftos(ij, fmat, 1,0);
        fmat ="MM%*.*lf";    maz= ftos(ii, fmat, 1,0); nmm[ii,ij] = maz $+ mbz;        
        fmat ="mu%*.*lf";    maz= ftos(ii, fmat, 1,0); nmu[ii,ij] = maz $+ mbz;        
        fmat ="e%*.*lf";     maz= ftos(ii, fmat, 1,0); nee[ii,ij] = maz $+ mbz;        
        fmat ="beta%*.*lf";  maz= ftos(ii, fmat, 1,0); nba[ii,ij] = maz $+ mbz;        
    ij=ij+1;
    endo;

    fmat ="xbeta%*.*lf"; nxb[ii,1]= ftos(ii, fmat, 1,0);
    fmat ="delta%*.*lf"; nde[ii+1,1] = ftos(ii, fmat, 1,0);
    fmat ="g%*.*lf";     ngs[ii,1]   = ftos(ii, fmat, 1,0);
ii=ii+1;
endo;

nab[1,1]= "notused";
nab[2,1]= "notused";
nab[3,1]= "notused";
nab[4,1]= "initPI*";
nab[5,1]= "initRHO*";
nab[6,1]= "notused";
nab[7,1]= "notused";

nde[1,1]= "delta0";

ii=1;
do while ii le maxnm+maxnos;
    ij=1;
    do while ij le maxnm+maxnos;
        fmat ="%*.*lf";  mbz= ftos(ij, fmat, 1,0);
        fmat ="h_%*.*lf"; maz= ftos(ii, fmat, 1,0); nhh[ii,ij] = maz $+ "," $+ mbz;        
		if nm le 9;
	        if ii eq ij; fmat ="h%*.*lf"; nhh[ii,ij]= ftos(ii, fmat, 1,0); endif;
		endif;
    ij=ij+1;
    endo;
ii=ii+1;
endo;
@-------------------------------------------------------------------------------------------------@

lnlik=0;
large =  nd*1e+10;

@-------------------------------------- READ INITIAL VALUES --------------------------------------@
bnew=zeros(kalln,1); 

if filebestb eq 1;
    load bnew[kalln,1] = ^file03;
elseif filebestb eq 0;
	bnew=vecr(baa)|vecr(bab)|vecr(bxb)|vecr(bde)|vecr(bhh)|vecr(bmm)|vecr(bmu)|vecr(bee)|vecr(bgs)|vecr(bba);
endif;

flagb = zeros(kalln,1); names = zeros(kalln,1); bound = zeros(kalln,2);

flagb=vecr(flaa)|vecr(flab)|vecr(flxb)|vecr(flde)|vecr(flhh)|vecr(flmm)|vecr(flmu)|vecr(flee)|vecr(flgs)|vecr(flba);
names=vecr(naa)|vecr(nab)|vecr(nxb)|vecr(nde)|vecr(nhh)|vecr(nmm)|vecr(nmu)|vecr(nee)|vecr(ngs)|vecr(nba);
bound[.,1]=vecr(lbaa)|lubab[.,1]|lubxb[.,1]|lubde[.,1]|vecr(lbhh)|vecr(lbmm)|vecr(lbmu)|vecr(lbee)|lubgs[.,1]|vecr(lbba);
bound[.,2]=vecr(ubaa)|lubab[.,2]|lubxb[.,2]|lubde[.,2]|vecr(ubhh)|vecr(ubmm)|vecr(ubmu)|vecr(ubee)|lubgs[.,2]|vecr(ubba);

kcml = sumc(flagb);

bout =   zeros(kalln,1);
astart = zeros(kcml,1);
lineb =  zeros(kcml,1);
SV = ones(kcml,1);

@-------------------------------------------------------------------------------------------------@

@-------------------------------------------- CML ------------------------------------------------@
_cml_Active = ones(kcml,1);
_cml_ParNames = zeros(kcml,1);
_ww_ = { -1e250 1e250 };                           
_cml_Bounds = ones(kcml,2).*_ww_;      
@----------------------------------- SELECT PARAMETERS FOR CML -----------------------------------@

lowa=zeros(kcml,1);
uppa=zeros(kcml,1);


ii=1;
iib=1;
do while iib le kalln;
    if flagb[iib] eq 1;
        astart[ii] =  bnew[iib];
        lineb[ii] = iib;
        _cml_Bounds[ii,1] =  bound[iib,1];
        _cml_Bounds[ii,2] =  bound[iib,2];
        _cml_ParNames[ii] = names[iib];

		lowa[ii,1] =  bound[iib,1];
		uppa[ii,1] =  bound[iib,2];

        ii=ii+1;
    endif;
iib=iib+1;
endo;

@----------------------------------- DEFINE SCALING VARIABLES ------------------------------------@

if scaleVAR eq 1;
    gg = 1;
    do while gg le kcml;
        test01:
        if (abs(astart[gg,1]) ge 1) and (abs(astart[gg,1]) ne 0);
            astart[gg,1] = astart[gg,1]/10;
            SV[gg,1] = SV[gg,1]/10;
            goto test01;
        endif;

        test02:
        if (abs(astart[gg,1]) lt 0.099999) and (abs(astart[gg,1]) ne 0);
            astart[gg,1] = astart[gg,1]*10;
            SV[gg,1] = SV[gg,1]*10;
            goto test02;
        endif;
    gg = gg+1;
    endo;
endif;
@-------------------------------------------------------------------------------------------------@

@--------------------------------------------- DATA ----------------------------------------------@    
zdata = zeros(ndtot,nmdata+nosdata); @ Data @
predhorizon = zeros(1,1);

mat = zeros(nm,1);

frequency = 4; @ No. of time periods in a year @
predHorizon = 1/4;

load zdata[ndtot,nmdata+nosdata] = ^file01;

znew = zeros(nd,nm+nos);

ij=1;
ii=1;
do while ii le nmdata+nosdata;
    if dataincl[ii,1] eq 1;
        znew[.,ij] = zdata[ndinit:ndlast,ii];
        if ii le nmdata;
            mat[ij,1] = matdata[ii,1];
        endif;
        ij=ij+1;
    endif;
ii=ii+1;
endo;

@znew[.,nmdata+1]=ln(znew[.,nmdata+1]);@


mat = mat/frequency;

/*
format /m1 /rd 12,8;
print "zdata" zdata; print;print;
print "znew"; print;
i=1;
do while i le nd;
    print i znew[i,1:nm+nos];
i=i+1;
endo;
print;
print "mat" mat;
stop;
*/

@--------------------------------------------------@
llim = zeros(kcml,1);
ulim = zeros(kcml,1);

i=1;
do while i le kcml;
    _cml_Bounds[i,1] = _cml_Bounds[i,1]*SV[i,1];
    _cml_Bounds[i,2] = _cml_Bounds[i,2]*SV[i,1];
    llim[i,1] = _cml_Bounds[i,1];
    ulim[i,1] = _cml_Bounds[i,2];
i=i+1;
endo;

@------------------------- Test whether initial values are within bounds -------------------------@
afix = zeros(1,1);

ii=1;
do while ii le kcml;
	if(astart[ii,1] lt _cml_Bounds[ii,1]) or (astart[ii,1] gt _cml_Bounds[ii,2]);
		if afix eq 0;
			print " Starting value outside bounds.";
		endif;

		format /m1 /rd 4,0;
		print "      Variable: " ii " = "  "lineb= " lineb[ii];
		format /m1 /rd 18,10;
		print "                           value " astart[ii,1];;
		print "     bounds " _cml_Bounds[ii,1] _cml_Bounds[ii,2];
		afix = 1;
	endif;
ii=ii+1;
endo;

if afix eq 1;
	print;
	print " -------------- Fix bounds a.u.b. -------------- ";
	stop;
endif;
@-------------------------------------------------------------------------------------------------@

@*********************************************************@
@*********        INITIALISING MATRICES        ***********@
@*********************************************************@

acml = zeros(kcml,1);
aopt = zeros(kcml,1);
aahelp = zeros(kcml,1);

AAA = zeros(nm+nos,1);
bestAAA = zeros(nm+nos,1);
beverAAA = zeros(nm+nos,1);

BBB = zeros(nm,ns);
BBBc = zeros(nm+nos,ns);
bestBBBc = zeros(nm+nos,ns);
beverBBBc = zeros(nm+nos,ns);
BBBg = zeros(nm+nos,ns);

besta = zeros(kcml,1);
bestb = zeros(kalln,1);
bestlnlik = zeros(1,1);
bestLRir = zeros(1,1);

bevera = zeros(kcml,1);
beverb = zeros(kalln,1);
beverlnlik = zeros(1,1);
beverLRir = zeros(1,1);

FF = zeros(nm+nos,nm+nos);
HH = zeros(nm+nos,nm+nos);
HHH = zeros(8,8);

LL = zeros(ns,ns);
LLL = zeros(7,7);
LLst = zeros(ns,ns);
LLste = zeros(ns,ns);

mt = zeros(1,1);

omega = zeros(ns,ns);
omegaVAR = zeros(ns,ns);

avphi = zeros(nm+nos,1);
avphicomp = zeros(ns, nm+nos);
psi = zeros(ns,1);
psist = zeros(ns,1);

QQ = zeros(ns,ns);
QQuncond = zeros(ns,ns);
QQQ = zeros(ns,ns);

teller03 = zeros(1,1);
teller04 = zeros(1,1);
teller08 = zeros(1,1);

perclarge = zeros(1,1);

V = zeros(ns,ns);
VV = zeros(ns,ns);

vaFF = zeros(nm+nos,1);
veFF = zeros(nm+nos,nm+nos);
varLL = zeros(ns,1);
vaiLL = zeros(ns,1);
verLL = zeros(ns,ns);
veiLL = zeros(ns,ns);

varLLst = zeros(ns,1);
vaiLLst = zeros(ns,1);
verLLst = zeros(ns,ns);
veiLLst = zeros(ns,ns);

vaV = zeros(ns,1);
veV = zeros(ns,ns);

wdata = zeros(nd,nm+nos);

X = zeros(ns,1);
XX = zeros(ns,1);
XXX = zeros(nd,ns);

XXpred = zeros(nd,ns);

ZZ = zeros(nm+nos,1);
ZZg = zeros(nm+nos,1);
@--------------------@

dt = zeros(1,1); 
step        = zeros(1,1); 
nsteps      = zeros(1,1); 

dt = 1/40; 
step        = dt;
nsteps      = 10/step+1;
precision = zeros(1,1); 
    precision = 10/dt;

CCC1        = zeros(1,ns+1);
DDD1        = zeros(1,ns+1);
eee1        = zeros(1,ns+1);
GGG1         = zeros(1,ns+1);
RK1            = zeros(nsteps,ns+1);

k1             = zeros(1,ns+1);
k2             = zeros(1,ns+1);
k3             = zeros(1,ns+1);
k4             = zeros(1,ns+1);
t             = zeros(nsteps,1);
@----------------------@

@--- Counters ---@
gg = zeros(1,1);
ii = zeros(1,1);
iii = zeros(1,1);
ij = zeros(1,1);
iij = zeros(1,1);
ik = zeros(1,1);
im = zeros(1,1);
iim = zeros(1,1);
in = zeros(1,1);
kkk =zeros(1,1);
kki =zeros(1,1);
pii = zeros(1,1);
@----------------@

Xstst =zeros(ns,1);
bestXstst =zeros(ns,1);
beverXstst =zeros(ns,1);

imagSpace=zeros(1,1);
bestSpace=zeros(1,1);
beverSpace=zeros(1,1);
everImag=zeros(1,1);
startlnlik=zeros(1,1);
LTir = zeros(1,1);
dglmbd = zeros(ns,ns);
RPcte = zeros(nm+nos,1);
RPdec = zeros(nd, (nm+nos)*ns +nm+nos);
counteigv=zeros(2,1);

statsystem=zeros(1,1);

@*********************************************************@
@******  FACKLER COND. MEAN/VARIANCE  ********************@
@*********************************************************@

@------ Fackler's matrix D -------@
DDf = zeros(ns^2,ns);

ii1 = zeros(1,1);
ii2 = zeros(1,1);

ii1 = 1;
do while ii1 le ns^2;
	ii2 = 1;
	do while ii2 le ns;
		if ii1 eq (ii2-1)*ns+ii2;
			DDf[ii1,ii2] = 1;
		endif;
	ii2=ii2+1;
	endo;
ii1=ii1+1;
endo;
@---------------------------------@

XXf = zeros(ns,1);
Vhelp = zeros(ns^2,1);
QQuncondf = zeros(ns,ns);
QQf = zeros(ns,ns);
atilde = zeros(ns+ns^2,1);
bigAtilde = zeros(ns+ns^2,ns+ns^2);

k1f0 = zeros(1,ns+ns^2);
k2f0 = zeros(1,ns+ns^2);
k3f0 = zeros(1,ns+ns^2);
k4f0 = zeros(1,ns+ns^2);

RKf0  = zeros(nsteps,ns+ns^2);
CCCf0 = zeros(1,ns+ns^2);
GGGf0 = zeros(1,ns+ns^2);
mf0	 = zeros(ns,1);
Vf0	 = zeros(ns^2,1);

k1f1	 = zeros(1,ns*(ns+ns^2));
k2f1 = zeros(1,ns*(ns+ns^2));
k3f1 = zeros(1,ns*(ns+ns^2));
k4f1 = zeros(1,ns*(ns+ns^2));

RKf1  = zeros(nsteps,ns*(ns+ns^2));
CCCf1 = zeros(1,ns*(ns+ns^2));
GGGf1 = zeros(1,ns*(ns+ns^2));
mf1	 = zeros(ns,ns);
Vf1	 = zeros(ns^2,ns);

helpf1 = zeros(ns+ns^2,1);
helpf2 = zeros(ns+ns^2,ns);

@*********************************************************@

bestlnlik     = -large;
beverlnlik    = -large;


@--- ANNEALING ----------------------------------------@
vm = scalevm*ones(kcml,1).*(uppa-lowa);

cvec = 2*ones(kcml,1);

if flagsimann eq 1;

screen on;
print;print;
format /m1 /rd 20,10;
print "Starting VM --------------------------  ";
print "vm " vm;
print;print;pause(0.5);

	call psimann(kcml,astart,lowa,uppa,nnt,frt,nns,neps,errps,maxevl,tpar,vm,cvec);
endif;

@print "----------> END Simulated Annealing procedure";@

if (flagsimann eq 1) and (continueopt eq 1);
	astart=aopt;
elseif (flagsimann eq 1) and (continueopt eq 0);
	print;print;print;print;print;
	stop;
endif;

@print "astart " aopt;@


@*********************************************************@
@***************     CML OPTIMISATION     ****************@
@*********************************************************@
format /m1 /rd 18,14;

teller03 = zeros(1,1);
teller04 = zeros(1,1);
teller08 = zeros(1,1);

print; print "KALMAN FILTER ESTIMATION - CML ALGORITHM "; pause(0.0001);

{aopt,bb,c,d,ret} = cml(znew,0,&mainj,astart);
{aopt,bb,c,d,ret} = cmlprt(aopt, bb, c, d, ret  );

@----- Print Factors ------@
call mainj(aopt,wdata);

format /m1 /rd 20,14;
print; 
print "-------------------";
print "XXX: Updated values" xxx;
@--------------------------@


gg = 1;
do while gg le kcml;
    aopt[gg,1] = aopt[gg,1]/SV[gg,1];
gg = gg+1; 
endo;          

ii=1;
iib=1;
do while iib le kalln;
    if flagb[iib] eq 1;
        bout[iib] =  aopt[ii];
        ii=ii+1;
    else;
        bout[iib] =  bnew[iib];
    endif;
iib=iib+1;
endo;

print; print;
print "Converged final values (bnew complete) !!! " bout;
print; print;

print "-------------------- END --------------------";
stop;
end;


@*********************************************************@
@***************      MAINJ-PROCEDURE      ***************@
@*********************************************************@

proc mainj(acml,wdata);

@---------------- Declare local variables ----------------@
LOCAL rowhelp, ndnow,
      vnam, m, bols, stb, vc, stderr, sigma, cx, rsq, resid, dwstat;
@---------------------------------------------------------@

rowhelp = zeros(1,1);
format /m1 /rd 16,14;

aahelp = acml.*(1/SV);

ii=1;
iib=1;
do while iib le kalln;
    if flagb[iib] eq 1;
        bnew[iib] =  aahelp[ii];
        ii=ii+1;
    endif;
iib=iib+1;
endo;

@--------------------------------------------------------------------------------@
baa=zeros(ns,4);  bab=zeros(7,1);     bxb=zeros(ns,1);  bde=zeros(1+ns,1);
bhh=zeros(maxnm+maxnos,maxnm+maxnos); bmm=zeros(ns,ns); bmu=zeros(ns,ns);
bee=zeros(ns,ns);                     bgs=zeros(ns,1);  bba=zeros(ns,ns);
@--- Form Matrices --------------------------------------------------------------@
baa = reshape(bnew[1:rows(baa)*cols(baa),1], rows(baa),cols(baa));
	rowhelp = rows(baa)*cols(baa);
bab = reshape(bnew[rowhelp+1:rowhelp+rows(bab)*cols(bab),1], rows(bab),cols(bab));
	rowhelp = rowhelp+rows(bab)*cols(bab);
bxb = reshape(bnew[rowhelp+1:rowhelp+rows(bxb)*cols(bxb),1], rows(bxb),cols(bxb));
	rowhelp = rowhelp+rows(bxb)*cols(bxb);
bde = reshape(bnew[rowhelp+1:rowhelp+rows(bde)*cols(bde),1], rows(bde),cols(bde));
	rowhelp = rowhelp+rows(bde)*cols(bde);
bhh = reshape(bnew[rowhelp+1:rowhelp+rows(bhh)*cols(bhh),1], rows(bhh),cols(bhh));
	rowhelp = rowhelp+rows(bhh)*cols(bhh);
bmm = reshape(bnew[rowhelp+1:rowhelp+rows(bmm)*cols(bmm),1], rows(bmm),cols(bmm));
	rowhelp = rowhelp+rows(bmm)*cols(bmm);
bmu = reshape(bnew[rowhelp+1:rowhelp+rows(bmu)*cols(bmu),1], rows(bmu),cols(bmu));
	rowhelp = rowhelp+rows(bmu)*cols(bmu);
bee = reshape(bnew[rowhelp+1:rowhelp+rows(bee)*cols(bee),1], rows(bee),cols(bee));
	rowhelp = rowhelp+rows(bee)*cols(bee);
bgs = reshape(bnew[rowhelp+1:rowhelp+rows(bgs)*cols(bgs),1], rows(bgs),cols(bgs));
	rowhelp = rowhelp+rows(bgs)*cols(bgs);
bba = reshape(bnew[rowhelp+1:rowhelp+rows(bba)*cols(bba),1], rows(bba),cols(bba));
@--------------------------------------------------------------------------------@


@--- INITIALIZING THE KALMAN FILTER ---@

AAA = zeros(nm+nos,1);
BBB = zeros(nm,ns);
D = zeros (ns,ns);
FF = zeros(nm+nos,nm+nos);
HH = zeros(nm+nos,nm+nos);
HHH = zeros(maxnm+maxnos,maxnm+maxnos);
LL = zeros(ns,ns);
LLL = zeros(maxns,maxns);
lnlik = zeros(1,1);
psi = zeros(ns,1);
psist = zeros(ns,1);
QQ = zeros(ns,ns);
V = zeros(ns,ns);
VV = zeros(ns,ns);
X = zeros(ns,1);
XX = zeros(ns,1);
XXX = zeros(nd,ns);
ZZ = zeros(nm+nos,1);
@--------------------------------------@

@--------------------------------@
bhh=lowmat(bhh);
bhh = bhh + bhh';
bhh = diagrv(bhh,0.5*diag(bhh));
@--------------------------------@

hh=bhh[1:nm+nos,1:nm+nos];

LLL[1,1] = bmm[1,1];
LLL[1,2] = bmm[1,2];
LLL[1,3] = bmm[1,3];
LLL[1,4] = -bmm[1,2];
LLL[1,5] = -bmm[1,3];

LLL[2,1] = bmm[2,1];
LLL[2,2] = bmm[2,2];
LLL[2,3] = bmm[2,3];
LLL[2,4] = -bmm[2,2];
LLL[2,5] = -bmm[2,3];

LLL[3,1] = bmm[3,1];
LLL[3,2] = bmm[3,2];
LLL[3,3] = bmm[3,3];
LLL[3,4] = -bmm[3,2];
LLL[3,5] = -bmm[3,3];

LLL[4,1] = 0.00;
LLL[4,2] = 0.00;
LLL[4,3] = 0.00;
LLL[4,4] = bmm[4,4];
LLL[4,5] = 0.00;

LLL[5,1] = 0.00;
LLL[5,2] = 0.00;
LLL[5,3] = 0.00;
LLL[5,4] = 0.00;
LLL[5,5] = bmm[5,5];

LL = LLL[1:ns,1:ns];
psi = -bmm*baa[.,2];

dglmbd=zeros(ns,ns);
psist = psi - diagrv(dglmbd, baa[1:ns,3])*baa[1:ns,4];
LLst  = LL  -(diagrv(dglmbd, baa[1:ns,3])*bba[1:ns,1:ns]+bMu[1:ns,1:ns]);

{varLL, vaiLL, verLL, veiLL} = eigrg2(LL);

if varLL lt 0;
	statsystem = 1;
else;
	statsystem = 0;
endif;

@ ------------------------------------ Test matrix LL ------------------------------------------ @
@ We need negative eigenvalues for stability, and non-zero determinant for invertibility  @

if notstable eq 0;
	if (varLL lt 0) and (det(LL) ne 0);
	else;

		lnlik = -large;

		if showMessage eq 1;
			print " matrix LL not negative definite " ;
			print "varLL" varLL; print;
			print "det(LL)" det(LL); print;
			pause(0.0001);
		endif;
		goto ssd; 
	endif; 

elseif notstable eq 1;

	if (varLL[1:3,1] lt 0);

	else;

		lnlik = -large;

		if showMessage eq 1;
			format 16,10;
			print "matrix LL not negative definite " ;
			print "varLL" varLL; print;
			print "verLL" verLL; print;
			pause(0.0001);
		endif;
		goto ssd; 
	endif; 

endif;

@---  To know if we are in the "imaginary space" or not ---@
imagSpace = 0;

if (vaiLL eq 0) and (veiLL eq 0);
else;
	imagSpace = 1;
endif; 
@----------------------------------------------------------@


@--- Unconditional MEAN of factors (stst: steady state) ----------@

if statsystem eq 1;
    X = -inv(LL)*psi;

elseif statsystem eq 0;

	x[1,1] = 0.0;
	x[2,1] = baa[4,2];
	x[3,1] = baa[5,2];
	x[4,1] = baa[4,2];
	x[5,1] = baa[5,2];
endif;

Xstst = X;
@-----------------------------------------------------------------@


@******  FACKLER COND. MEAN/VARIANCE ********@

@if notstable eq 0;@
if statsystem eq 1;
	@---- Unconditional Variance -----@
	Vhelp = -inv(LL.*.eye(ns)+eye(ns).*.LL)*(bee.*.bee)*DDf*(baa[.,4]-bba*inv(LL)*psi);
	QQuncondf = reshape(Vhelp,ns,ns);
	V = QQuncondf;
	@---------------------------------@

	if diag(V) ge 0;
	else;
		lnlik = -large;

		if showMessage eq 1;
			format 16,10;
			print "Unconditional Variance matrix with negative diagonal" ;
			print "V" v;
			pause(0.0001);
		endif;
		goto ssd; 
	endif;

endif;

atilde = zeros(ns+ns^2,1);
bigAtilde = zeros(ns+ns^2,ns+ns^2);

atilde = psi|((bee.*.bee)*DDf*baa[.,4]);	

bigAtilde[1:ns,1:ns] = LL; 
bigAtilde[ns+1:ns+ns^2,1:ns] = (bee.*.bee)*DDf*bba; 
bigAtilde[ns+1:ns+ns^2,ns+1:ns+ns^2] = LL.*.eye(ns)+eye(ns).*.LL; 

step = predHorizon/20; 
nsteps = predHorizon/step;

call M0V0(atilde,bigAtilde);
call M1V1(atilde,bigAtilde);

@*********************************************************@

@-----------------------------------------@
call RUNGE1(bnew);
BBBc[1:nm,1:ns]=BBB;
BBBc[nm+1:nm+nos,1:nos] = eye(nos);
@-----------------------------------------@

X = Xstst;

@ ------------------------------ START TIME LOOP ------------------------------  @
i = 1;
do while i le nd;

	@--- FACKLER COND. MEAN/VARIANCE ------------@
	helpf1 = (mf0|Vf0)+(mf1|Vf1)*X;
	XXf = helpf1[1:ns,1];
	QQf = reshape(helpf1[ns+1:ns+ns^2,1], ns, ns);
	@--------------------------------------------@

	XX = XXf;
	QQ = QQf;
	omega = mf1;

    @------ Forecast VARIANCE --------------@    

        V = omega*V*omega' + QQ;

    @---------------------------------------@

	@--- Test whether diagonal of V is positive ---@
	if diag(V) ge 0;
	else;
		lnlik = -large;

		if showMessage eq 1;
			format 16,10;
			print "Forecast Variance matrix with negative diagonal" ;
			print "V" v;
			pause(0.0001);
		endif;
		goto ssd; 
	endif;
	@----------------------------------------------@

    @---- TEST matrix V --------------------------------------@
    @--- V has to be positive definite (all eigenvalues gt 0) --@

        {vaV,veV} = eigv(V);

		if vaV gt 0;

		else;
			lnlik = -large;

			if showMessage eq 1;
				print "Matrix V not positive definite ";  
				print "vaV computed in time loop  " i "  " vaV;
				pause(0.0001);
			endif;
			goto ssd; 
		endif; 
    @ ---------------------------------------------------------------@


    @----- Pricing errors -----@

        ZZ = znew[i,.]' - AAA - BBBc*XX;

    @----- Variance of Pricing errors ------------------------------@

        FF = BBBc*V*(BBBc)' + HH;

    @---- TEST matrix FF --------------------------------------@
    @--- FF has to be positive definite (all eigenvalues gt 0) --@

        {vaFF,veFF} = eigv(FF);

        if abs(det(FF)) lt 1E-150;
            lnlik = -large;
            if showMessage eq 1;
                print " det(FF) is basically ZERO ! ";  
                format /m1 /re 14,6;
                print "                                 " det(ff);
                pause(0.01);
            endif;
            goto ssd; 
        endif; 

		if vaFF gt 0;
		else;

			lnlik = -large;

			if showMessage eq 1;
				print "Matrix FF not positive definite ";  
				print " vaFF computed in time loop  " i "  " vaff;
				pause(0.0001);
			endif;
			goto ssd; 
		endif; 
    @ ---------------------------------------------------------------@

    @--- Updating Mean (X) ---------------------@

        X = XX + V*(BBBc)'*inv(FF)*ZZ ; 

    @--- Updating Variance ---------------------@

        VV = V - V*(BBBc)'*inv(FF)*BBBc*V;
        V = VV;

	@--- Make elements very close to zero equal to zero ---@

	v=substute(v, abs(v).lt 10e-10, zeros(ns,ns));

	@------------------------------------------------------@

	@--- Test whether diagonal of V is positive ---@
	if diag(V[3:5,3:5]) ge 0;

	else;

		lnlik = -large;

		if showMessage eq 1;
			format /m1 /rd 14,6;
			print "Updating Variance matrix with negative diagonal" ;
			print "V" v;
			pause(0.0001);
		endif;
		goto ssd; 
	endif;
	@----------------------------------------------@

    @---- TEST matrix V --------------------------------------@
    @--- V has to be positive definite (all eigenvalues gt 0) --@

        {vaV,veV} = eigv(V);

		if (vaV[1:3] gt 0) and (vaV[1:3] ge 0);
		else;
			lnlik = -large;

			if showMessage eq 1;
				print "Updating Variance matrix V not positive definite ";  
				format /m1 /re 14,10;
				print "vaV(eigenvalues) computed in time loop  " i "  " vaV;
				pause(0.0001);
			endif;
			goto ssd; 
		endif; 
    @ ---------------------------------------------------------------@

    @--- Loglikelihood -------------------------@

	lnlik = lnlik - 0.5 *( ln(det(FF)) + ZZ'*inv(FF)*ZZ);

	@---------- SAVE TO PRINT AFTERWARDS -------------@
	@--- XXX = Update value,   XX = Forecast value ---@
	XXX[i,.] = X';   
	XXpred[i,.]=XX';
    @-------------------------------------------------@
i = i+1;
endo;
@-------------------------------- END TIME LOOP ------------------------------- @
ssd:

teller03 = teller03 + 1;

if teller03 eq 1;
    startlnlik = lnlik;
endif;

if lnlik eq -large;
    teller08 = teller08 + 1;
endif;

perclarge = teller08/teller03*100;

@--------------------------------------- SAVE BEST RESULTS ---------------------------------------@
if (lnlik gt bestlnlik) or ((bestlnlik eq -large) and (lnlik lt -large));

	if diag(bhh) ge 0;
	else;
		goto dontsave;
	endif;

    bestlnlik = lnlik;
    besta = acml;
    bestb = bnew;
    bestXstst = Xstst;
    bestSpace = imagSpace;
    if imagSpace eq 1;
        everImag = 1;
    endif;

	bestLRir = bde[1] + bde[2:1+ns]'*Xstst;
endif;

if lnlik gt beverlnlik;
    beverlnlik=lnlik;
    bevera =acml;
    beverb =bnew;
    beverAAA =AAA;
    beverBBBc =BBBc;
    beverXstst = Xstst;
    beverLRir = bestLRir;
    beverSpace = imagSpace;
endif;
@-------------------------------------------------------------------------------------------------@
dontsave:
@-------------------------------------- PRINT BEST RESULTS ---------------------------------------@

if ((teller03 ge teller04+printCML) or (teller03 eq 1)) and (lnlik ne -large);
    screen off;

    if (teller03 ge printCML);
        format /ma1 /rd 20,14;
            output file = ^file03 reset;
                print beverb;
            output file = ^file03 off;
    endif;

	output file = ^file02 reset;

	call print02(beverlnlik,beverb,beverAAA,beverBBBc,beverXstst,beverLRir,beverSpace);
	print;print;
	FORMAT /M1 /RD 4,0;        
	print " Teller03 PROC J=" teller03 " lnlik=-large= " teller08 "or" perclarge " %";;
	print " everImag=" everImag " imag= " bestSpace;

	call print01(beverlnlik,beverb,beverAAA,beverBBBc,beverXstst,beverLRir,beverSpace);

	output file = ^file02 off;
    screen on;        
endif;

@-------------------------------------------------------------------------------------------------@

@--- Show on the SCREEN --------------------------------------------------------------------------@
if (printCML gt 0);
	if ((teller03 ge teller04+printCML) or (teller03 eq 1)) and (lnlik ne -large);
		cls;

		FORMAT /M1 /RD 4,0;        
		print "teller03 PROC J=" teller03 " lnlik=-large= " teller08 "or" perclarge " %  everImag=" everImag " imag= " bestSpace;
		call print01(beverlnlik,beverb,beverAAA,beverBBBc,beverXstst,beverLRir,beverSpace);
		if teller03 gt 1; teller04 = teller04 + printCML; endif;
	endif;    
endif;

@-------------------------------------------------------------------------------------------------@
endlnlik:



retp(lnlik);   
endp;


@*********************************************************@
@***************       RISK PREMIUM       ****************@
@*********************************************************@

proc(2)=riskp(BBB,BBBc,baa,bmu,bee,bba);

@---------------- Declare local variables ----------------@
LOCAL Ehelp, iir, help1, help2, empriskp, iis,
vnam, m, bols, stb, vc, stderr, sigma, cx, rsq, resid, dwstat;
@---------------------------------------------------------@

RPcte = zeros(nm+nos,1);
Ehelp = zeros(nm+nos,ns);
RPdec = zeros(nd, (nm+nos)*ns +nm+nos);


@--- RISK PREMIUM DECOMPOSITION  -----------@
@--- compute the risk premium per annum ----@

if nos eq 0;
	RPcte = -(mat.*BBB)*((baa[.,3].*baa[.,4]));
	Ehelp = -(mat.*BBB)*((baa[.,3].*bba+bmu));
else;
	RPcte = -((mat|ones(2,1)).*(BBB|BBBc[nm+1:nm+nos,.]))*((baa[.,3].*baa[.,4]));
	Ehelp = -((mat|ones(2,1)).*(BBB|BBBc[nm+1:nm+nos,.]))*((baa[1:ns,3].*bba+bmu));
endif;


iir=1;
do while iir le nm+nos;
	RPdec[.,1+(iir-1)*ns:iir*ns]= (Ehelp[iir,.]'.*XXx')';
iir=iir+1;
endo;

@--- total ---@
RPdec[.,(nm+nos)*ns+1:(nm+nos)*ns+nm+nos] = (RPcte.*ones(nm+nos,nd))' + (Ehelp*XXx')';

retp(rpcte,rpdec);   
endp;


@*********************************************************@
@**********             RUNGE 1                 **********@
@*************    Computes AAA's and BBB's    ************@
@*********************************************************@

proc(2) = RUNGE1(bnew);

AAA = zeros(nm+nos,1);
BBB = zeros(nm,ns);
CCC1 = zeros(1,ns+1);
kkk = zeros(1,1);

step = zeros(1,1); 
step = 1/20; 

nsteps = zeros(1,1); 
nsteps = 10/step;

RK1 = zeros(nsteps,ns+1);
RK1[1,.] = CCC1;

ii = 1;
do while ii le nsteps;
    call HOM1(CCC1);
    RK1[ii,.] = CCC1;
ii = ii + 1;
endo;


kkk = 1; 
do while kkk le nm; 
    AAA[kkk,1] = -RK1[mat[kkk]/step,1]/mat[kkk]; 

    kki=1;
    do while kki le ns;
        BBB[kkk,kki] = -RK1[mat[kkk]/step,kki+1]/mat[kkk];
    kki=kki+1;
    endo;
kkk = kkk + 1; 
endo;

retp(AAA,BBB);
endp;

@*************************************@
proc(1) = HOM1(DDD1);

    CCC1 = zeros(1,ns+1);

    k1 = step*DER_HOM1(DDD1);
    k2 = step*DER_HOM1(DDD1 +0.5*k1);
    k3 = step*DER_HOM1(DDD1 +0.5*k2);
    k4 = step*DER_HOM1(DDD1 +k3);

    CCC1 = DDD1 + k1/6 + k2/3 + k3/3 + k4/6;

retp(CCC1);
endp;


@*************************************@
proc(1) = DER_HOM1(EEE1);

GGG1 = zeros(1,ns+1);


@--- AAAs ---@
iii=1;
do while iii le ns;
	GGG1[1,1] = GGG1[1,1] + psist[iii,1]*EEE1[1,iii+1] + 0.5*EEE1[1,iii+1]^2*baa[iii,4];
iii=iii+1;
endo;

@--- inclusion of alpha0 in int. rate equation ---@
GGG1[1,1] = GGG1[1,1] - bde[1];


@--- BBBs ---@
iii=1;
do while iii le ns;
	iij=1;
	do while iij le ns;
		GGG1[1,iii+1] = GGG1[1,iii+1] + LLst[iij,iii]*EEE1[1,iij+1] + 0.5*EEE1[1,iij+1]^2*bba[iij,iii];
	iij=iij+1;
	endo;

	GGG1[1,iii+1] = GGG1[1,iii+1] -bde[iii+1];
iii=iii+1;
endo;
@-----------------------------------------------------------@

retp(GGG1);
endp;


@********************************************@
@*******     RUNGE-KUTTA (M0V0)       *******@
@********************************************@
proc(2) = M0V0(atilde,bigAtilde);

	mf0	 = zeros(ns,1);
	Vf0	 = zeros(ns^2,1);
	RKf0 = zeros(nsteps,ns+ns^2);
	CCCf0 = zeros(1,ns+ns^2);

	ii = 1;
	do while ii le nsteps;
		call HOMf0(CCCf0);
		RKf0[ii,.] = CCCf0;
	ii = ii + 1;
	endo;
	
	mf0 = RKf0[(predhorizon)/step,1:ns]'; 
	Vf0 = RKf0[(predhorizon)/step,ns+1:ns+ns^2]'; 

retp(mf0,Vf0);
endp;

@*************************************@
proc(1) = HOMf0(DDDf0);

	CCCf0 = zeros(1,ns+ns^2);

	k1f0 = step*DERf0(DDDf0);
	k2f0 = step*DERf0(DDDf0+0.5*k1f0);
	k3f0 = step*DERf0(DDDf0+0.5*k2f0);
	k4f0 = step*DERf0(DDDf0+k3f0);

	CCCf0 = DDDf0 + k1f0/6 + k2f0/3 + k3f0/3 + k4f0/6;

retp(CCCf0);
endp;

@*************************************@
proc(1) = DERf0(EEEf0);

	GGGf0[1,.] = (atilde+bigAtilde*EEEf0[1,.]')';

retp(GGGf0);
endp;


@********************************************@
@*******     RUNGE-KUTTA (M1V1)       *******@
@********************************************@
proc(2) = M1V1(atilde,bigAtilde);

	mf1 = zeros(ns,ns);
	Vf1 = zeros(ns^2,ns);

	RKf1 = zeros(nsteps,ns*(ns+ns^2));

	CCCf1 = zeros(1,ns*(ns+ns^2));
	CCCf1 = vec(eye(ns)|zeros(ns^2,ns))';

	ii = 1;
	do while ii le nsteps;
		call HOMf1(CCCf1);
		RKf1[ii,.] = CCCf1;
	ii = ii + 1;
	endo;

	helpf2 = reshape(RKf1[(predhorizon)/step,.]',ns,ns+ns^2)';
	mf1 = helpf2[1:ns,1:ns]; 
	Vf1 = helpf2[ns+1:ns+ns^2,1:ns]; 

retp(mf1,Vf1);
endp;

@*************************************@
proc(1) = HOMf1(DDDf1);

	k1f1 = step*DERf1(DDDf1);
	k2f1 = step*DERf1(DDDf1+0.5*k1f1);
	k3f1 = step*DERf1(DDDf1+0.5*k2f1);
	k4f1 = step*DERf1(DDDf1+k3f1);

	CCCf1 = DDDf1 + k1f1/6 + k2f1/3 + k3f1/3 + k4f1/6;

retp(CCCf1);
endp;

@*************************************@
proc(1) = DERf1(EEEf1);

	GGGf1[1,.] = (vec(bigAtilde*reshape(EEEf1,ns,ns+ns^2)'))';

retp(GGGf1);
endp;


@*********************************************************@
@***************     print01: IN PROC J    ***************@
@*********************************************************@

proc print01(bestlnlik,bestb,bestAAA,bestBBBc,bestXstst, bestLRir,bestSpace);

@---------------- Declare local variables ----------------@
LOCAL rowhelp, pij ;
@---------------------------------------------------------@

baa = reshape(bestb[1:rows(baa)*cols(baa),1], rows(baa),cols(baa));
	rowhelp = rows(baa)*cols(baa);
bab = reshape(bestb[rowhelp+1:rowhelp+rows(bab)*cols(bab),1], rows(bab),cols(bab));
	rowhelp = rowhelp+rows(bab)*cols(bab);
bxb = reshape(bestb[rowhelp+1:rowhelp+rows(bxb)*cols(bxb),1], rows(bxb),cols(bxb));
	rowhelp = rowhelp+rows(bxb)*cols(bxb);
bde = reshape(bestb[rowhelp+1:rowhelp+rows(bde)*cols(bde),1], rows(bde),cols(bde));
	rowhelp = rowhelp+rows(bde)*cols(bde);
bhh = reshape(bestb[rowhelp+1:rowhelp+rows(bhh)*cols(bhh),1], rows(bhh),cols(bhh));
	rowhelp = rowhelp+rows(bhh)*cols(bhh);
bmm = reshape(bestb[rowhelp+1:rowhelp+rows(bmm)*cols(bmm),1], rows(bmm),cols(bmm));
	rowhelp = rowhelp+rows(bmm)*cols(bmm);
bmu = reshape(bestb[rowhelp+1:rowhelp+rows(bmu)*cols(bmu),1], rows(bmu),cols(bmu));
	rowhelp = rowhelp+rows(bmu)*cols(bmu);
bee = reshape(bestb[rowhelp+1:rowhelp+rows(bee)*cols(bee),1], rows(bee),cols(bee));
	rowhelp = rowhelp+rows(bee)*cols(bee);
bgs = reshape(bestb[rowhelp+1:rowhelp+rows(bgs)*cols(bgs),1], rows(bgs),cols(bgs));
	rowhelp = rowhelp+rows(bgs)*cols(bgs);
bba = reshape(bestb[rowhelp+1:rowhelp+rows(bba)*cols(bba),1], rows(bba),cols(bba));

@---------------------------------@
bhh=lowmat(bhh);
bhh = bhh + bhh';
bhh = diagrv(bhh,0.5*diag(bhh));
@---------------------------------@


format 16,10;
print;
print "K         " bmm; print;
print "baa  ";
print "                      cte              lambda           sigma" baa; print;
print "ksi (market price of risk)" bmu; print;


print "R (measurement error covariance matrix) ";

pii=1;
do while pii le nm+nos;
	pij=1;
	do while pij le nm+nos;
		if bhh[pii,pij] ne 0;
			format 2,0;
		    print "bhh[" pii "," pij "]=";; format 18,14; bhh[pii,pij];
		endif;
	pij=pij+1;
	endo;
pii=pii+1;
endo;


print "--------------------------------------------------------------------------------------------------------";
FORMAT /M1 /RD 14,10;
print "     AAA          BBBc";

pii=1;
do while pii le nm+nos;
    print bestAAA[pii,1] bestBBBc[pii,.];
pii=pii+1;
endo;

FORMAT /M1 /RD 2,0;
print "             ";;
iim=1; 
do while iim le nm+nos; 
	print "   avphi" iim;;
iim=iim+1;
endo;
print;
FORMAT /M1 /RD 10,6;

@----------------------------------------- Risk Premium ------------------------------------------@

call riskp(BBB,BBBc,baa,bmu,bee,bba);
avphi= meanc(RPdec[.,(nm+nos)*ns+1:(nm+nos)*ns+nm+nos]);

iim=1; 
do while iim le nm; 
	avphicomp[.,iim] = 100*meanc(abs(RPdec[.,1+(iim-1)*ns:iim*ns]))/sumc(meanc(abs(RPdec[.,1+(iim-1)*ns:iim*ns])));
iim=iim+1;
endo;


@-------------------------------------------------------------------------------------------------@
FORMAT /M1 /RD 10,6;
print "     Best    " avphi';
FORMAT /M1 /RD 10,2;

pii=1;
do while pii le ns;
	print  " fac  (%)" avphicomp[pii,.];
pii=pii+1;
endo;


FORMAT /M1 /RD 13,10;

print " Loglik        " lnlik     "       lnlik/nd   " lnlik/nd;
print "      Best     " bestlnlik "           Best   " bestlnlik/nd;

print "                       Long-term interest rate   " bestLRir ;
format 6,0;
print "teller03 PROC J " teller03;; format 3,0;
print "               everImag  " everImag "      imag  " bestSpace  ;;
print "      No. var. in CML " kcml;
print "--------------------------------------------------------------------------------------------------------";
print;print;

pause(0.0001);

retp(mt);
endp;

@*********************************************************@
@***************     print02: IN PROC J    ***************@
@*********************************************************@

proc print02(bestlnlik,bestb,bestAAA,bestBBBc,bestXstst,bestLRir,bestSpace);

@---------------- Declare local variables ----------------@
LOCAL rowhelp, pij;
@---------------------------------------------------------@

baa = reshape(bestb[1:rows(baa)*cols(baa),1], rows(baa),cols(baa));
	rowhelp = rows(baa)*cols(baa);
bab = reshape(bestb[rowhelp+1:rowhelp+rows(bab)*cols(bab),1], rows(bab),cols(bab));
	rowhelp = rowhelp+rows(bab)*cols(bab);
bxb = reshape(bestb[rowhelp+1:rowhelp+rows(bxb)*cols(bxb),1], rows(bxb),cols(bxb));
	rowhelp = rowhelp+rows(bxb)*cols(bxb);
bde = reshape(bestb[rowhelp+1:rowhelp+rows(bde)*cols(bde),1], rows(bde),cols(bde));
	rowhelp = rowhelp+rows(bde)*cols(bde);
bhh = reshape(bestb[rowhelp+1:rowhelp+rows(bhh)*cols(bhh),1], rows(bhh),cols(bhh));
	rowhelp = rowhelp+rows(bhh)*cols(bhh);
bmm = reshape(bestb[rowhelp+1:rowhelp+rows(bmm)*cols(bmm),1], rows(bmm),cols(bmm));
	rowhelp = rowhelp+rows(bmm)*cols(bmm);
bmu = reshape(bestb[rowhelp+1:rowhelp+rows(bmu)*cols(bmu),1], rows(bmu),cols(bmu));
	rowhelp = rowhelp+rows(bmu)*cols(bmu);
bee = reshape(bestb[rowhelp+1:rowhelp+rows(bee)*cols(bee),1], rows(bee),cols(bee));
	rowhelp = rowhelp+rows(bee)*cols(bee);
bgs = reshape(bestb[rowhelp+1:rowhelp+rows(bgs)*cols(bgs),1], rows(bgs),cols(bgs));
	rowhelp = rowhelp+rows(bgs)*cols(bgs);
bba = reshape(bestb[rowhelp+1:rowhelp+rows(bba)*cols(bba),1], rows(bba),cols(bba));

@---- We need this since we only optimize the lower diagonal of HH ----@
bhh=lowmat(bhh);
bhh = bhh + bhh';
bhh = diagrv(bhh,0.5*diag(bhh));
@----------------------------------------------------------------------@


FORMAT /M1 /RD 16,10;
print "beverlnlik/nd  " beverlnlik/nd;
print;
print $filedir;
print;

FORMAT /M1 /RD 6,0;
print "   No. states (ns)             " ns;
print "   No. observable states (nos) " nos;
print "   No. maturities (nm)         " nm;

print "   Interval (data)             " interval " = " interval+(interval-1)*2  "  months"; 
print "   ndTot                       " ndTot;
print "   ndInit                      " ndInit;
print "   ndLast                      " ndLast;
print "   matdata                     " matdata';
print "   dataIncl                    " dataIncl';
print "   No. maturities              " nm;
print "----------------------------------------------------------------------";

FORMAT /M1 /RD 18,14;
print "REAL beverb !!!!! Copy to bestbnew.txt." bestb;
print;print;print;

print "--- REAL beverb !!!!! Copy to beginning of the program -------------- ";
print;
print "@bestEVERlnlik/nd = " beverlnlik/nd "  @";
print "@" $filedir "@";
print;
format /m1 /rdn 20,14;
print "@     Kappa               Theta                Lambda               c                    d  @";
print "    let baa["  ;; format 1,0; print ns ",4]= "  ;; format /m1 /rd 20,14; print baa ";"; print;

print "@   Delta0  @";
print "@   Delta1            Delta2             Delta3             Delta4             Delta5             Delta6             Delta7@";
print "    let bde[";; format /rdn 1,0; print ns+1 ",1]= "  ; format /m1 /rd 18,14; print bde[1]; print bde[2:ns+1]' ";"; print;

format /m1 /rd 20,14;
print "@ h1                 h2                 h3                 h4                 h5                 h6                 h7                 h8@";


pii=1;
do while pii le nm+nos;
	pij=1;
	do while pij le nm+nos;
		if bhh[pii,pij] ne 0;
			format 2,0;
		    print "bhh[" pii "," pij "]=";; format 18,14; bhh[pii,pij] ";" ;
		endif;
	pij=pij+1;
	endo;
pii=pii+1;
endo;

print "@  M1                 M2                 M3                 M4                 M5                 M6                 M7@";
print "  let bmm["    ;; format /rdn 1,0; print ns "," ns "]= ";; format /m1 /rd 18,14; print bmm ";"; print;

print "@  Mu1                Mu2                Mu3                Mu4                Mu5                Mu6                Mu7@";
print "  let bmu["    ;; format /rdn 1,0; print ns "," ns "]= ";; format /m1 /rd 18,14; print bmu ";"; print;


print "--- copy until here --------------------------------------------------";

FORMAT /M1 /RD 18,12;
print;
format /m1 /rd 16,10;
print "---------------------------------------------------------------------------------";
print "      Variable        Initial Value        Value              Change";
print "---------------------------------------------------------------------------------";                    
print "          lnlik/nd       "   startlnlik/nd "  " beverlnlik/nd "    ";; format /m1 /rd 18,14; beverlnlik/nd-startlnlik/nd;
print "---------------------------------------------------------------------------------";    

ii=1;
iib=1;
do while iib le kalln;
	if flagb[iib] eq 1;
    	format /m1 /rd 3,0;
        print  ii;;
    	format /m1 /rd 5,0;
        print  iib;;
        format /m1 /rd 10,0;
        print $names[iib];; 
        format /m1 /rd 16,10;
        print "  " astart[ii]/sv[ii]  "  " beverb[iib] "    ";; format /m1 /rd 18,14; beverb[iib]-astart[ii]/sv[ii];  
		ii=ii+1;
	endif;
iib=iib+1;
endo;
print "---------------------------------------------------------------------------------";    

pause(0.0001);

retp(mt);
endp;




/*--------------------------------------------------------------------------

Procedure SIMULATED ANNEALING

translated to Gauss from the FORTRAN code provided by
William L. Goffe - http://wuecon.wustl.edu/~goffe/

based on paper by:
Goffe, Ferrier, and Rogers (1994), "Global optimization of statistical
functions with simulated annealing", J.Econometrics 60, 65-99.

--------------------------------------------------------------------------*/

proc psimann(kcml,aact,lowa,uppa,nnt,frt,nns,neps,errps,maxevl,tpar,vm,cvec);

/*-------------------------------------------------------------------------
INPUT:
kcml   : (1,1)    no. of variables to be maximized
aact   : (kcml,1) starting values and afterwards actual values
lowa   : (kcml,1) lower bounds
uppa   : (kcml,1) upper bounds
nnt    : (1,1)    no. of iterations before reduction in parameter T (tpar)
frt    : (1,1)    reduction factor for parameter T (tpar)
nns    : (1,1)    no. of iterations before step length (VM) is adjusted
neps   : (1,1)    no. of final function values to decide upon termination
errps  : (1,1)    error tolerance for termination
maxevl : (1,1)    maximum no. of function evaluations
tpar   : (1,1)    parameter T
vm     : (kcml,1) step length for each variable
cvec   : (kcml,1) vector that controls the step lenght adjustment

OUTPUT:
aopt   : (kcml,1) optimum values
-------------------------------------------------------------------------*/

LOCAL	aact,
		nfcnev, nacp,
		LLact, LLopt, LLtrial, LLprev, 
		iit, iis, iin,
		atrial, pval, aratio,
		nacc, nup, ndown, nrej, nnew, nobds;


nfcnev =zeros(1,1);     @ total no. of evaluations             @
nacp   =zeros(kcml,1);  @ acceptance for each variable @

nacc   =zeros(1,1);     @ no. of accepted function evaluations @
nup    =zeros(1,1);     @ no.of ups         @
ndown  =zeros(1,1);     @ no. of downs      @
nrej   =zeros(1,1);     @ no. of rejections @
nnew   =zeros(1,1);     @ no.of new??       @
nobds  =zeros(1,1);     @ no. of evaluations outside boudns    @



LLact   =zeros(1,1);            @ optimum lnlik                  @
LLopt   =zeros(1,1);            @ optimum lnlik                  @
LLtrial =zeros(1,1);            @ trial value for lnlik          @
LLprev  = -large*ones(neps,1);  @ vector with last optimum lnlik @

atrial =zeros(kcml,1);  @ trial values for vector of variables @

pval    =zeros(1,1);            @ value of p, eq.(2) @
aratio  =zeros(1,1);            @ acceptance ration @


@--- stop if parameter T is negative ---@

if tpar le 0;
	print;print;print;print;print;print;
	print "------ parameter T (tpar) must be positive";
	print;print;print;print;print;print;
	stop;
endif;
@---------------------------------------@

aopt=aact;

call mainj(aact,znew);
LLact = lnlik;
LLopt = lnlik;
LLprev[1,1] = lnlik;


simannagain:


nup    =zeros(1,1);
ndown  =zeros(1,1);
nrej   =zeros(1,1);
nnew   =zeros(1,1);
nobds  =zeros(1,1);


iit=1;
do while iit le nnt;

	iis=1;
	do while iis le nns;

		iin=1;
		do while iin le kcml;
			atrial =  aact;

			@--- select trial value ---@
			atrial[iin,1] = aact[iin,1] + (-1 + rndu(1,1)) *vm[iin,1];
			@--------------------------@

			@--- if trial value is outside bounds, select a point inside the bounds ---@
			if (atrial[iin,1] lt lowa[iin,1]) or (atrial[iin,1] gt uppa[iin,1]);
				atrial[iin,1] = lowa[iin,1] + (uppa[iin,1]-lowa[iin,1])*rndu(1,1);
				nobds=nobds+1;
			endif;
			@--------------------------------------------------------------------------@

			call mainj(atrial,znew);
			LLtrial = lnlik;

			nfcnev=nfcnev+1;

			@--- if maximum no. of evaluations is reached, terminate ---@
			if nfcnev ge maxevl;
				retp(aopt);
			endif;
			@-----------------------------------------------------------@

			@--- accept new point if lnlik increased ----------@

			if LLtrial ge LLact;
				aact = atrial;
				LLact = LLtrial;

				nacc=nacc+1;
				nacp[iin,1]=nacp[iin,1]+1;
				nup=nup+1;

				if LLtrial ge LLopt;
					aopt = atrial;
					LLopt = LLtrial;
					nnew=nnew+1;
				endif;


			@--- use Metropolis criteria if lnlik decreased ---@
			elseif LLtrial lt LLact;
				pval = exp((LLtrial-LLact)/tpar);

				if  pval ge (rndu(1,1)); 
					aact = atrial;
					LLact = LLtrial;
					nacc=nacc+1;
					nacp[iin,1]=nacp[iin,1]+1;
					ndown=ndown+1;
				else;
					nrej=nrej+1;
				endif;
			endif;
			@--------------------------------------------------@

		iin=iin+1;
		endo;

screen on;
format /m1 /rd 4,0;
print "iit nnt " iit " of" nnt "  until reduce T  ";
print "iis nns " iis " of" nns "  until adjust VM ";
format /m1 /rd 16,10;
print "LLopt/nd~LLact/nd                     " LLopt/nd~LLact/nd;
format /m1 /rd 5,0;
print "total nobds  nacc   nup ndown  nrej  nnew";
print iis*kcml~nobds~nacc~nup~ndown~nrej~nnew;
print;print;
pause(0.1);

	iis=iis+1;
	endo;

	@--- Adjust VM so that approximately half of all evaluations are accepted ---@

	iin=1;
	do while iin le kcml;

		aratio = nacp[iin,1]/nns;

		if aratio ge 0.6;
			vm[iin,1] = vm[iin,1]*(1+ cvec[iin,1]*(aratio-0.6)/0.4);
		elseif aratio lt 0.4;
			vm[iin,1] = vm[iin,1]/(1+ cvec[iin,1]*(0.4-aratio)/0.4);
		endif;

		if vm[iin,1] gt (uppa[iin,1]-lowa[iin,1]);
			vm[iin,1] = (uppa[iin,1]-lowa[iin,1]);
		endif;


	iin=iin+1;
	endo;
	@----------------------------------------------------------------------------@

screen on;
print;print;
format /m1 /rd 20,10;
print "iit -------- " iit " of " nnt;
print "nacp~vm~aopt " nacp~vm~aopt;
print "LLopt/nd~LLact/nd " LLopt/nd~LLact/nd;
print;pause(0.01);

	nacp = zeros(kcml,1);

iit=iit+1;
endo;


@--- Check termination criteria ---@

@--- terminate ---@
if (LLopt-LLprev[1,1] le errps) and (abs(LLact*ones(neps,1)-LLprev) le errps);
	retp(aopt);
endif;

print "LLact LLprev difference " LLact*ones(neps,1)~LLprev~abs(LLact*ones(neps,1)-LLprev);
print;print;pause(0.05);


LLprev[2:neps,1] = LLprev[1:neps-1,1];
LLprev[1,1] = LLact;


@--- continue ---@

tpar = frt*tpar;
LLact=LLopt;
aact=aopt;

goto simannagain;

@----------------------------------@

retp(aopt);
endp;
